#!/usr/bin/perl -w
use strict;
use Socket;
use MIME::Base64;

use XML::Parser;
use Data::Dumper;

use sigtrap;
use Time::HiRes qw ( time sleep );
use POSIX qw(:sys_wait_h);


my ($CGServerAddress,$CGServerLogin,$CGServerPassword);
my ($domainName,$nUsers,$handleErrors,$scenarioFile);
my @theScenario;
my $debug=0;
my $port = 11024;

readConfigFile();
readScenarioFile();

my ($data,$errCode);
my $xmlParser = new XML::Parser(Style => 'Tree');

$errCode=XIMSS_Connect($CGServerAddress,$port);
if($errCode) {
  print "Unable to connect to CGPro XIMSS: $errCode\n";
  exit(1);
}

if($errCode=checkError($data=XIMSS_Login($CGServerLogin,$CGServerPassword))) {
  die "Login failed: $errCode\n";
}
print "Creating test accounts...\n";

$errCode=createAccounts();
XIMSS_Logout();
if($errCode) {
  die "$errCode\n";
}

$debug=0;

print "Launching $nUsers user processes...\n";

my $countChildren=$nUsers;

my $t0=Time::HiRes::time();

for(my $idx=1;$idx<=$nUsers;$idx++) {
  my $pid=fork();
  if(defined $pid) {
    unless($pid) { # child
      stressUser($idx);
      exit(0);
    } else {
      #print "pid=$pid\n" 
    }
  } else {
    print "idx=$idx fork() failed: $!\n";
  }
}

print "Waiting for children processes...\n";
while(wait()!=-1) {};

  my $t1=Time::HiRes::time();
  printf("\nElapsed time: %.3f s.\n",$t1-$t0);

exit(0);

sub stressUser {
  my $accountName=sprintf("user%05d",$_[0]);
  $accountName.='@'.$domainName;
  my $lineCnt=0;
  foreach(@theScenario) {
    my $line=$_; $lineCnt++;
    print "Line: $line\n" if($debug);
    next if(/^\s*$/);
    next if(/^#/);
    if(/^\</) {
      if($errCode=checkError(XIMSS_raw_command($line))) {
        print "$accountName Error executing $line: $errCode\n";
      } 
      
    }elsif(/^cmd\s+(.*)/i) {
      my $command=$1;
      my ($params,$body);
      if($command=~/(.*)\/(.*)/) {
        $command=$1; $body=$2;
      }
      if($command=~/^(\S+)\s+(.*)/) {
        $command=$1; $params=$2;
      }
      if($errCode=checkError(XIMSS_command($command,$params,$body))) {
        $params='' unless($params);
        print "$accountName Error executing $command $params: $errCode\n";
      } 
      
    } elsif(/^connect/i) {
      $errCode=XIMSS_Connect($CGServerAddress,$port);
      if($errCode) {
        print "$accountName Unable to connect to CGPro XIMSS: $errCode\n";
        exit(1);
      }
    } elsif(/^login/i) {
      if($errCode=checkError($data=XIMSS_Login($accountName,'abc'))) {
        die "$accountName login failed: $errCode\n";
      }
    } elsif(/^logout/i) {
      XIMSS_Logout();

    } elsif(/^pause\s+(\+?\d+)/i) {
      my $delay=$1;
      if($delay=~/^\+(\d+)/) {
        $delay=int(rand($1+1));
      }
      sleep($delay);   

    } elsif(/^writeSiteFile\s+(.*)/i) {
      my ($fName,$chunkSize)=($1,undef);
      if($fName=~/(.*)\s+(\d+.*)/) {
        $fName=$1; $chunkSize=$2;
        $chunkSize=$1*1024 if($chunkSize=~/(\d+)K/i);
      } else {
        $chunkSize=30*1024;
      }
      writeSiteFile($fName,$chunkSize);

    } elsif(/^readSiteFile\s+(.*)/i) {
      my ($fName,$chunkSize)=($1,undef);
      if($fName=~/(.*)\s+(\d+.*)/) {
        $fName=$1; $chunkSize=$2;
        $chunkSize=$1*1024 if($chunkSize=~/(\d+)K/i);
      }
      readSiteFile($fName,$chunkSize);

    } elsif(/^appendMessage\s+(.*)/i) {
      my ($fName,$boxName)=($1,'INBOX');
      if($fName=~/(\S+)\s+(.*)/) {
        $fName=$1; $boxName=$2;
        $boxName=$1 if($boxName=~/\"(.*)\"/); #"
      }
      appendMessage($fName,$boxName);
    } elsif(/^submitMessage\s+(.*)/i) {
      my ($fName,$address)=($1,'');
      $address=sprintf("user%05d",1+int(rand($nUsers)));
      $address.='@'.$domainName;
      submitMessage($fName,$address);

    } elsif(/^openMailbox\s+(.*)/i) {
      openMailbox($1);
    } elsif(/^closeMailbox\s+(.*)/i) {
      closeMailbox($1);
    } elsif(/^readMailbox\s+(.*)/i) {
      readMailbox($1);
    } elsif(/^readAllMailboxes/i) {
      readAllMailboxes();

    }else{
      die "Line $lineCnt Syntax error: $line\n";
    }
  }
  exit(0);

}

sub readMessageFile {
  my ($fileName)=@_;
  my @header;
  my @addrHeaders=qw/From To Cc Bcc Return-Path Sender Reply-To Disposition-Notification-To Recent-From Recent-To Return-Receipt-To Errors-To/;
  my @dateHeaders=qw/Date Resent-Date/;
  unless(open(FILE,$fileName)) {
    print "Can't open $fileName: $!\n";
    return undef;
  }
  my $curLine='';
  while(<FILE>) {
    chomp;
    tr/\r\n//;
    #print "line=$_\n";
    if($curLine) {
      if(/^\s+(.*)/) {
        
        $curLine.=' '.$1;
      } else {
        $curLine=~/^(\S*)\:\s*(.*)/;
        my ($hdrName,$hdrData)=($1,$2);
        my $auxData='';
        foreach(@dateHeaders) {
          if(/$hdrName$/i) {
            $auxData=' timeShift="0" localTime="20070101T180030"'; #we don't parde dates
            $hdrData='20070101T180030';
            last;
          }
        }
        foreach(@addrHeaders) {
          if(/$hdrName$/i) {
            $hdrData=$1 if($hdrData=~/<(.*)>/); #we don't save real name part
            last;
          }
        }
        $hdrData=~s/&/&amp;/g;
        $hdrData=~s/</&lt;/g;
        $hdrData=~s/>/&gt;/g;
        push(@header,"<$hdrName$auxData>$hdrData</$hdrName>");

        $curLine=$_;
      }
    }else{
      $curLine=$_;
    }
    last unless($_);
  }
  my $msgBody='';
  while(<FILE>) {
    chomp;
    $msgBody.=$_."\012";
  }
  close(FILE);

  $msgBody=~s/&/&amp;/g;
  $msgBody=~s/</&lt;/g;
  $msgBody=~s/>/&gt;/g;

  my $data="<EMail>".join('',@header).'<MIME type="text" subtype="plain">'.$msgBody.'</MIME></EMail>';

}

sub appendMessage {
  my ($fileName,$boxName)=@_;
  my $msgData=readMessageFile($fileName);
  $data=XIMSS_command('messageAppend',{targetMailbox=>$boxName,mailboxClass=>' '},$msgData);
  if($errCode=checkError($data)) {
    print "Failed to append $fileName file: $errCode\n";
    return;
  }
}

sub submitMessage {
  my ($fileName,$accountName)=@_;
  my $msgData=readMessageFile($fileName);
  $data=XIMSS_command('messageSubmit',{targetMailbox=>"Sent Items",mailboxClass=>' '},'<Envelope-To>'.$accountName.'</Envelope-To>'.$msgData);
  if($errCode=checkError($data)) {
    print "Failed to submit $fileName file: $errCode\n";
    return;
  }
}


sub writeSiteFile {
  my ($fileName,$chunkSize)=@_;
  $chunkSize=30*1024 unless($chunkSize);
  unless(open(FILE,$fileName)) {
    print "Can't open $fileName: $!\n";
    return;
  }
  my $cnt=0;
  my $buf;
  $data=XIMSS_command('fileWrite',{fileName=>$fileName},'<base64></base64>');
  binmode(FILE);
  while (read(FILE, $buf, $chunkSize)) {
    my $fData=encode_base64($buf,'');
    $data=XIMSS_command('fileWrite',{fileName=>$fileName,position=>'append'},'<base64>'.$fData.'</base64>');
  }
  close(FILE);
}

sub readSiteFile {
  my ($fileName,$chunkSize)=@_;
  $chunkSize=1024*1024 unless($chunkSize);
  my $pos=0;
  my $totalSize=1;
  
  while($pos<$totalSize) {
    $data=XIMSS_command_Err('fileRead',{fileName=>$fileName,type=>'binary',position=>$pos,limit=>$chunkSize});
    if($errCode=checkError($data)) {
      print "Failed to read $fileName site file: $errCode\n";
      return;
    }
    if($totalSize==1) { 
      foreach(@$data) {
        my $item=$_;
        if($item->[0] eq 'fileData') {
          $item=$item->[1]->[0];
          $totalSize=$item->{size};
          #print "item=".Dumper($item); 
        }
      }
    }
    $pos+=$chunkSize;
  }
}

sub openMailbox {
  my $mailboxName=$_[0];
  $data=XIMSS_command('folderOpen',{folder=>$mailboxName,sortField=>"INTERNALDATE",sortOrder=>"asc"},"<field>Subject</field><field>UID</field>");
}

sub closeMailbox {
  my $mailboxName=$_[0];
  $data=XIMSS_command('folderClose',{folder=>$mailboxName});
}

sub readMailbox {
  my $mailboxName=$_[0];
#  print "Reading mailbox: $mailboxName\n"; 
  my $data=XIMSS_command_Err('folderBrowse ',{folder=>$mailboxName,indexFrom=>"0", indexTill=>"999"});
  if($errCode=checkError($data)) {
    print "Failed to browse a folder: $errCode\n";
    return;
  }

  my @mailboxList=();
  foreach(@$data) {
    my $item=$_;
    if($item->[0] eq 'folderReport') {
      $item=$item->[1]->[0];
      push(@mailboxList,$item->{'UID'});
      #print "item=".Dumper($item); 
    }
  }
  
 # $data=XIMSS_command('folderRead ',{folder=>$mailboxName,totalSizeLimit=>1000000,UID=>$mailboxList[0]});

  foreach(@mailboxList) {
    my $messageUID=$_;
    $data=XIMSS_command('folderRead ',{folder=>$mailboxName,totalSizeLimit=>1000000,UID=>$messageUID});
  }
}

sub readAllMailboxes {
  my $data=XIMSS_command_Err('mailboxList');
  if($errCode=checkError($data)) {
    print "Failed to list mailboxes: $errCode\n";
    return;
  }
  my @mailboxList=();
  foreach(@$data) {
    my $item=$_;
    if($item->[0] eq 'mailbox') {
      $item=$item->[1]->[0];
      unless($item->{'pureFolder'} && $item->{'pureFolder'} eq 'YES') {
        push(@mailboxList,$item->{'mailbox'});
        #print "item=".Dumper($item);
      }   
    }
  }
  foreach(@mailboxList) {
    if($errCode=checkError(openMailbox($_))) {
      print "Failed to open mailbox $_: $errCode\n";
    } else {
      readMailbox($_);
    }  
  }
}


sub readConfigFile {
  my $cfgFileName='ximssStressTest.cfg';
  open(FILE,$cfgFileName) || die "Can't open $cfgFileName: $!";
  $handleErrors='Y';
  while(<FILE>) {
    next if(/^#/);
    $CGServerAddress=$1 if(/hostname\s+(.*)/i);
    $CGServerLogin=$1 if(/adminLogin\s+(.*)/i);
    $CGServerPassword=$1 if(/adminPAssword\s+(.*)/i);
  
    $domainName=$1 if(/TestDomain\s+(.*)/i);
    $nUsers=$1 if(/NUsers\s+(.*)/i);
    $handleErrors=$1 if(/HandleErrors\s+(.*)/i);
    $scenarioFile=$1 if(/ScenarioFile\s+(.*)/i);
    
  }
  close(FILE);
  if($handleErrors=~/^Y/i) {
    $handleErrors=1;
  }elsif($handleErrors=~/^N/i) {
    $handleErrors=0;
  }else{
    die "Invalid value for HandleErrors: $handleErrors\n";
  }  
  
  die "Hostname is not defined in $cfgFileName\n" unless($CGServerAddress);
  die "AdminLogin is not defined in $cfgFileName\n" unless($CGServerLogin);
  die "AdminPassword is not defined in $cfgFileName\n" unless($CGServerPassword);
  
  die "TestDomain is not defined in $cfgFileName\n" unless($domainName);
  die "NUsers is not defined in $cfgFileName\n" unless($nUsers);
  die "ScenarioFile is not defined in $cfgFileName\n" unless($scenarioFile);
  
  #print "a=$CGServerAddress, l=$CGServerLogin p=$CGServerPassword\n";
  #print "a=$domainName, l=$nUsers p=$scenarioFile\n";

}

sub readScenarioFile {
  open(FILE,$scenarioFile) || die "Can't open $scenarioFile: $!";
  while(<FILE>) {
    chomp;
    push(@theScenario,$_);
  }
  close(FILE);
}


sub createAccounts {
  $data=XIMSS_command_Err('cliExecute',undef,"Route LoginPage\@$domainName");
  my $objectExists=0;
  foreach(@$data) {
    my $item=$_;
    if($item->[0] eq 'cliResult') {
      $item=$item->[1]->[2];
      $objectExists=1 if($item=~/LOCAL/);
      last;
    }
  }
  if(!$objectExists && ($errCode=checkError($data=XIMSS_command_Err('cliExecute',undef,"CreateDomain $domainName")))) {
    return "Can't create domain $domainName: $errCode";
  }

  for(my $idx=1;$idx<=$nUsers;$idx++) {
    my $accountName=sprintf("user%05d",$idx);
    $accountName.='@'.$domainName;
    if($errCode=checkError($data=XIMSS_command_Err('cliExecute',undef,"Route $accountName"))) {
      if($errCode=checkError($data=XIMSS_command_Err('cliExecute',undef,"CreateAccount $accountName {Password=abc;}"))) {
        return "Can't create account $accountName: $errCode";
      }
    }
    
  }
  undef;
}


#---------------------
sub XIMSS_Connect {
  my ($address,$port)=@_;
  $port=11024 unless($port);
  
  my $iaddr   = inet_aton($address);
  unless($iaddr) {
    return "no host: $address";
  }  
  my $paddr   = sockaddr_in($port, $iaddr);

  my $proto   = getprotobyname('tcp');
  unless(socket(SOCK, PF_INET, SOCK_STREAM, $proto)) {
    return "socket error: $!";
  }  
  unless(connect(SOCK, $paddr)) {
    return "connect error: $!";
  }
  undef;
}

sub XIMSS_Login {
  my ($login,$password)=@_;
  return XIMSS_command_Err('login',{authData=>$login,password=>$password});
}

sub XIMSS_Logout {
  return XIMSS_command('bye');
}

sub XIMSS_command {
  my ($command, $paramsRef,$body)=($_[0],$_[1],$_[2]);
  if($handleErrors) {
    return XIMSS_command_Err($command, $paramsRef,$body);
  } else {
    return XIMSS_command_Noerr($command, $paramsRef,$body);
  }
}

my $idCounter=0;
sub XIMSS_command_Err {
  my ($command, $paramsRef,$body)=($_[0],$_[1],$_[2]);

  my $id='a'.$idCounter++;
  my $cmd="<$command id=\"$id\"";
  if($paramsRef) {
    if(ref($paramsRef) eq 'HASH') {
      foreach(keys(%$paramsRef)) {
        if(@$paramsRef{$_}) {
          $cmd.=" $_=\"@$paramsRef{$_}\"";
        } else {
          $cmd.=" $_=\"\"";
        }  
      }
    } else {
      $cmd.=' '.$paramsRef;
    }
  }  
  if($body) {
    $cmd.=">$body</$command>";
   } else {  
    $cmd.='/>';
  }
  sendCommand($cmd);
  my @answer;
  my $line='';
  while($line = readLine()) {
    print "response: $line\n" if($debug);
    my $xmlHash=$xmlParser->parse($line);
    push(@answer,$xmlHash);
    #print Dumper($xmlHash);
    last if($line =~ /^<response id=\"$id\"/ || $line =~ /^<error /); # "
  }
#print "answer=".Dumper(@answer);  
  return \@answer;
}
sub XIMSS_command_Noerr {
  my ($command, $paramsRef,$body)=($_[0],$_[1],$_[2]);
  my $id='a'.$idCounter++;
  my $cmd="<$command id=\"$id\"";
  if($paramsRef) {
    if(ref($paramsRef) eq 'HASH') {
      foreach(keys(%$paramsRef)) {
        if(@$paramsRef{$_}) {
          $cmd.=" $_=\"@$paramsRef{$_}\"";
        } else {
          $cmd.=" $_=\"\"";
        }  
      }
    } else {
      $cmd.=' '.$paramsRef;
    }
  }  
  if($body) {
    $cmd.=">$body</$command>";
   } else {  
    $cmd.='/>';
  }
  sendCommand($cmd);

  my @answer=();
  
  my $line='';
  while($line = readLine()) {
    #print "Nresponse: $line\n" if($debug);
    #print Dumper($xmlHash);
    last if($line =~ /^<response id=\"$id\"/ || $line =~ /^<error /); # "
  }
  return \@answer;
}
 
sub XIMSS_raw_command {
  my ($command)=@_;
  my $id;
  if($command=~/id=\"(.*)\"/) {
    $id=$1;
  }else{
    die "the 'id' is missing in $command\n";
  }
  sendCommand($command);
#  while(!($line=readResponse())) { sleep(1);};
  my @answer;
  
  my $line='';
  while($line = readLine()) {
    print "response: $line\n" if($debug);
    if($handleErrors) {
      my $xmlHash=$xmlParser->parse($line);
      push(@answer,$xmlHash);
      #print Dumper($xmlHash);
    }  
    last if($line =~ /^<response id=\"$id\"/ || $line =~ /^<error /); # "
  }
#print "answer=".Dumper(@answer);  
  return \@answer;
}
  
 
sub checkError {
  my $dataRef=$_[0];
  if(ref($dataRef) eq 'ARRAY') {
   #print "size=".scalar(@$dataRef)."\n";
  
    my $lastItem=$dataRef->[scalar(@$dataRef)-1];
    if(ref($lastItem) eq 'ARRAY') {
      $lastItem=$lastItem->[1]->[0];
       #print "ref=".ref($lastItem)."\n";      
      if(ref($lastItem) eq 'HASH') {
        my $eText=@$lastItem{'errorText'};
        return $eText if($eText);
      }
    }
  };
  undef;
} 
 


    
sub sendCommand {
  my ($command)=@_;
  print "sending: $command\n" if($debug);
  send(SOCK,$command ."\0",0);
}
    
sub isReadable {
 my $rin = '';
 vec($rin, fileno(SOCK), 1) = 1;
 return select( $rin, undef, undef, 0);
}
    
sub readLine {
#  return undef unless(isReadable);
  my $ln='';
  for(;;) {
    my $ch;
    recv(SOCK,$ch,1,0);
    last if($ch eq "\0");
    $ln.=$ch;
  }
  return $ln;
}




__END__
